<?php
/**
 * Plugin Name: WooCommerce Smatpay Gateway
 * Requires Plugins: woocommerce
 * Plugin URI: https://developer.smatpay.africa/plugins/woocommerce-smatpay-payment-gateway/
 * Description: Receive payments using the Zimbabwe Smatpay payments provider.
 * Author: Smatpay Developers
 * Author URI: https://developer.smatpay.africa/
 * Version: 1.0.0
 * Requires at least: 6.4
 * Tested up to: 6.6
 * WC requires at least: 9.1
 * WC tested up to: 9.3
 * Requires PHP: 7.4
 * PHP tested up to: 8.3
 * text-domain: smatpay-woo
 *
 * @package WooCommerce Gateway Smatpay
 */

defined( 'ABSPATH' ) || exit;

define( 'WC_GATEWAY_SMATPAY_VERSION', '1.0.0' );
define( 'WC_GATEWAY_SMATPAY_URL', untrailingslashit( plugins_url( '/', __FILE__ ) ) );
define( 'WC_GATEWAY_SMATPAY_PATH', untrailingslashit( plugin_dir_path( __FILE__ ) ) );
define('SMATPAY_GET_CURRENCY_URL', 'https://live.smatpay.africa/init/currencies/all');
define('PROD_SMATPAY_GET_TOKEN_URL', 'https://live.smatpay.africa/auth/authenticate/merchant');
define('PROD_SMATPAY_GET_PAYMENT_LINK_URL', 'https://live.smatpay.africa/api/merchant/fastCheckOut');
define('SANDBOX_SMATPAY_GET_TOKEN_URL', 'https://live.smatpay.africa/auth/authenticate/sandbox/merchant');
define('SANDBOX_SMATPAY_GET_PAYMENT_LINK_URL', 'https://live.smatpay.africa/api/merchant/sandbox/fastCheckOut');

// Load Composer autoloader
if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    require_once __DIR__ . '/vendor/autoload.php';
}
/**
 * Initialize the gateway.
 *
 * @since 1.0.0
 */
function woocommerce_smatpay_init() {
	if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
        die("Please install Woocommerce");
		return;
	}

	require_once plugin_basename( 'includes/WC_Gateway_Smatpay.php' );
	require_once plugin_basename( 'includes/WC_Gateway_PayFast_Privacy.php' );
	load_plugin_textdomain( 'woocommerce-gateway-smatpay', false, trailingslashit( dirname( plugin_basename( __FILE__ ) ) ) );

    add_filter( 'woocommerce_payment_gateways', 'woocommerce_smatpay_add_gateway' );
}
add_action( 'plugins_loaded', 'woocommerce_smatpay_init', 11 );

/**
 * Add links to the plugin action links.
 *
 * @since 1.4.13
 *
 * @param array $links Plugin action links.
 * @return array Modified plugin action links.
 */
function woocommerce_smatpay_plugin_links( $links ) {
	$settings_url = add_query_arg(
		array(
			'page'    => 'wc-settings',
			'tab'     => 'checkout',
			'section' => 'wc_gateway_smatpay',
		),
		admin_url( 'admin.php' )
	);

	$plugin_links = array(
		'<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'Settings', 'woocommerce-gateway-smatpay' ) . '</a>',
		'<a href="https://www.woocommerce.com/my-account/tickets/">' . esc_html__( 'Support', 'woocommerce-gateway-smatpay' ) . '</a>',
		'<a href="https://docs.woocommerce.com/document/smatpay-payment-gateway/">' . esc_html__( 'Docs', 'woocommerce-gateway-smatpay' ) . '</a>',
	);

	return array_merge( $plugin_links, $links );
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'woocommerce_smatpay_plugin_links' );

/**
 * Add the smatpay gateway to WooCommerce
 *
 * @since 1.0.0
 *
 * @param string[] $gateways
 * @return string[]
 */
function woocommerce_smatpay_add_gateway( $gateways ) {
	$gateways[] = 'WC_Gateway_Smatpay';
	return $gateways;
}

add_action( 'woocommerce_blocks_loaded', 'woocommerce_smatpay_woocommerce_blocks_support' );

/**
 * Add the gateway to WooCommerce Blocks.
 *
 * @since 1.4.19
 */
function woocommerce_smatpay_woocommerce_blocks_support() {
	if ( class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
		require_once WC_GATEWAY_SMATPAY_PATH . '/includes/WC_Smatpay_Blocks_Support.php';
		add_action(
			'woocommerce_blocks_payment_method_type_registration',
			function ( Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry ) {
				$payment_method_registry->register( new WC_Smatpay_Blocks_Support() );
			}
		);
	}
}

/**
 * Declares compatibility with Woocommerce features.
 *
 * List of features:
 * - custom_order_tables
 * - product_block_editor
 *
 * @since 1.6.1 Rename function
 * @return void
 */
function woocommerce_smatpay_declare_feature_compatibility() {
	if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
			'custom_order_tables',
			__FILE__
		);

		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
			'product_block_editor',
			__FILE__
		);
	}
}
add_action( 'before_woocommerce_init', 'woocommerce_smatpay_declare_feature_compatibility' );

/**
 * Display notice if WooCommerce is not installed.
 *
 * @since 1.5.8
 */
function woocommerce_smatpay_missing_wc_notice() {
	if ( class_exists( 'WooCommerce' ) ) {
		// Display nothing if WooCommerce is installed and activated.
		return;
	}

	echo '<div class="error"><p><strong>';
	printf(
		/* translators: %s WooCommerce download URL link. */
		esc_html__( 'WooCommerce Smatpay Gateway requires WooCommerce to be installed and active. You can download %s here.', 'woocommerce-gateway-smatpay' ),
		'<a href="https://woocommerce.com/" target="_blank">WooCommerce</a>'
	);
	echo '</strong></p></div>';
}
add_action( 'admin_notices', 'woocommerce_smatpay_missing_wc_notice' );

function smatpay_redirect_woo_checkout( $order_id ){
    $order = wc_get_order( $order_id );

    $payment_method = $order->get_payment_method();

    if ($payment_method === 'smatpay') {
        $order->reduce_order_stock();
        $order->update_status('processing', 'Automatically changed to processing.');
    }
}

add_action( 'woocommerce_thankyou', 'smatpay_redirect_woo_checkout');
