<?php

use Smatpay\Constants\TestCredentials;
use Smatpay\Definitions\AuthenticationBuilder;
use Smatpay\Definitions\FastCheckoutBuilder;
use Smatpay\Gateway\FastCheckOut;

class WC_Gateway_Smatpay extends WC_Payment_Gateway {

	public $version;

	/**
	 * Data to send to Smatpay.
	 *
	 * @var array $data_to_send
	 */
	protected $data_to_send = array();

	/**
	 * Merchant ID.
	 *
	 * @var string $merchant_id
	 */
	protected $merchant_id;

	/**
	 * Merchant Key.
	 *
	 * @var string $merchant_key
	 */
	protected $merchant_key;

    /**
     * Merchant API Key.
     *
     * @var string $merchant_api_key
     */
    protected $merchant_api_key;

    /**
     * Payer Account ID.
     *
     * @var string $payer_account_id
     */
    protected $payer_account_id;

    /**
     * Profile ID.
     *
     * @var string $profile_id
     */
    protected $profile_id;

	/**
	 * Smatpay URL.
	 *
	 * @var string $url
	 */
	protected $url;

	/**
	 * Smatpay Validate URL.
	 *
	 * @var string $validate_url
	 */
	protected $validate_url;

	/**
	 * Response URL.
	 *
	 * @var string $response_url
	 */
	protected $response_url;

	/**
	 * Send debug email.
	 *
	 * @var bool $send_debug_email
	 */
	protected $send_debug_email;

	/**
	 * Debug email.
	 *
	 * @var string $debug_email
	 */
	protected $debug_email;

    protected $test_mode;

	/**
	 * Enable logging.
	 *
	 * @var bool $enable_logging
	 */
	protected $enable_logging;

	/**
	 * Available countries.
	 *
	 * @var array $available_countries
	 */
	protected $available_countries;

	/**
	 * Available currencies.
	 *
	 * @var array $available_currencies
	 */
	protected $available_currencies;

	/**
	 * Logger instance.
	 *
	 * @var WC_Logger $logger
	 */
	protected $logger;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->version      = WC_GATEWAY_SMATPAY_VERSION;
		$this->id           = 'smatpay';
		$this->method_title = __( 'Smatpay', 'woocommerce-gateway-smatpay' );
		/* translators: 1: a href link 2: closing href */
		$this->method_description  = sprintf( __( 'Smatpay operates by redirecting the user to its platform, where they can securely enter their payment details.', 'woocommerce-gateway-smatpay' ), '<a href="https://smatpay.io/">', '</a>' );
		$this->icon                = WP_PLUGIN_URL . '/' . plugin_basename( dirname( __DIR__ ) ) . '/assets/images/icon.png';
		$this->debug_email         = get_option( 'admin_email' );
		$this->available_countries = array( 'ZW');

		$this->available_currencies = (array) apply_filters( 'woocommerce_gateway_smatpay_available_currencies', array( 'ZAR', 'USD', 'ZIG' ) );

		// Supported functionality.
		$this->supports = array(
			'products',
			'subscriptions',
			'subscription_cancellation',
			'subscription_suspension',
			'subscription_reactivation',
			'subscription_amount_changes',
			'subscription_date_changes',
			'subscription_payment_method_change',
			'subscription_payment_method_change_customer',
		);

		$this->init_form_fields();
		$this->init_settings();

		if ( ! is_admin() ) {
			$this->setup_constants();
		}

		// Setup default merchant data.
		$this->merchant_id      = $this->get_option( 'merchant_id' );
		$this->merchant_key     = $this->get_option( 'merchant_key' );
        $this->merchant_api_key = $this->get_option( 'merchant_api_key' );
        $this->payer_account_id = $this->get_option( 'payer_account_id' );
        $this->profile_id       = $this->get_option( 'profile_id' );
        $this->test_mode        = $this->get_option('testmode');
		$this->url              = 'https://dev-payments.smatpay.africa/process?aff=woo-free';
		$this->validate_url     = 'https://dev-payments.smatpay.africa/query/validate';
		$this->title            = $this->get_option( 'title' );
		$this->response_url     = add_query_arg( 'wc-api', 'WC_Gateway_Smatpay', home_url( '/' ) );
		$this->send_debug_email = false; //'yes' === $this->get_option( 'send_debug_email' );
		$this->description      = $this->get_option( 'description' );
		$this->enabled          = 'yes' === $this->get_option( 'enabled' ) ? 'yes' : 'no';
		$this->enable_logging   = false; //'yes' === $this->get_option( 'enable_logging' );

		// Setup the test data, if in test mode.
		if ( 'yes' === $this->get_option( 'testmode' ) ) {
            $this->url              = 'https://dev-payments.smatpay.africa/process?aff=woo-free&sandbox=true';
            $this->validate_url     = 'https://dev-payments.smatpay.africa/query/validate?sandbox=true';
			$this->add_testmode_admin_settings_notice();
		} else {
			$this->send_debug_email = false;
		}

		add_action( 'woocommerce_api_wc_gateway_smatpay', array( $this, 'check_itn_response' ) );
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		add_action( 'woocommerce_receipt_smatpay', array( $this, 'receipt_page' ) );
		add_action( 'woocommerce_scheduled_subscription_payment_' . $this->id, array( $this, 'scheduled_subscription_payment' ), 10, 2 );
		add_action( 'woocommerce_subscription_status_cancelled', array( $this, 'cancel_subscription_listener' ) );
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );

		// Add fees to order.
		add_action( 'woocommerce_admin_order_totals_after_total', array( $this, 'display_order_fee' ) );
		add_action( 'woocommerce_admin_order_totals_after_total', array( $this, 'display_order_net' ), 20 );

		// Change Payment Method actions.
		add_action( 'woocommerce_subscription_payment_method_updated_from_' . $this->id, array( $this, 'maybe_cancel_subscription_token' ), 10, 2 );

		// Add support for WooPayments multi-currency.
		add_filter( 'woocommerce_currency', array( $this, 'filter_currency' ) );

		add_filter( 'nocache_headers', array( $this, 'no_store_cache_headers' ) );
	}

	/**
	 * Use the no-store, private cache directive on the order-pay endpoint.
	 *
	 * This prevents the browser caching the page even when the visitor has clicked
	 * the back button. This is required to determine if a user has pressed back while
	 * in the smatpay gateway.
	 *
	 * @since 1.6.2
	 *
	 * @param string[] $headers Array of caching headers.
	 * @return string[] Modified caching headers.
	 */
	public function no_store_cache_headers( $headers ) {
		if ( ! is_wc_endpoint_url( 'order-pay' ) ) {
			return $headers;
		}

		$headers['Cache-Control'] = 'no-cache, must-revalidate, max-age=0, no-store, private';
		return $headers;
	}

	/**
	 * Initialise Gateway Settings Form Fields
	 *
	 * @since 1.0.0
	 */
	public function init_form_fields() {
		$this->form_fields = array(
			'enabled'          => array(
				'title'       => __( 'Enable/Disable', 'woocommerce-gateway-smatpay' ),
				'label'       => __( 'Enable Or Disable', 'woocommerce-gateway-smatpay' ),
				'type'        => 'checkbox',
				'description' => __( 'This controls whether or not this gateway is enabled within WooCommerce.', 'woocommerce-gateway-smatpay' ),
				'default'     => 'no',
				'desc_tip'    => true,
			),
			'title'            => array(
				'title'       => __( 'Smatpay Woo Payment Gateway', 'woocommerce-gateway-smatpay' ),
				'type'        => 'text',
				'description' => __( 'Show title in the checkout page.', 'woocommerce-gateway-smatpay' ),
				'default'     => __( 'Smatpay Woo Payment Gateway', 'woocommerce-gateway-smatpay' ),
				'desc_tip'    => true,
			),
			'description'      => array(
				'title'       => __( 'Description', 'woocommerce-gateway-smatpay' ),
				'type'        => 'textarea',
				'description' => __( 'Please remit your payment to the shop to allow for the delivery to be made', 'woocommerce-gateway-smatpay' ),
				'default'     => '',
				'desc_tip'    => true,
			),
            'instructions'      => array(
                'title'       => __( 'Instructions', 'woocommerce-gateway-smatpay' ),
                'type'        => 'textarea',
                'description' => __( 'Instructions to the customer', 'woocommerce-gateway-smatpay' ),
                'default'     => '',
                'desc_tip'    => true,
            ),
			'testmode'         => array(
				'title'       => __( 'Smatpay Sandbox', 'woocommerce-gateway-smatpay' ),
				'type'        => 'checkbox',
				'description' => __( 'Place the payment gateway in development mode.', 'woocommerce-gateway-smatpay' ),
				'default'     => 'yes',
			),
			'merchant_id'      => array(
				'title'       => __( 'Merchant ID', 'woocommerce-gateway-smatpay' ),
				'type'        => 'text',
				'description' => __( '', 'woocommerce-gateway-smatpay' ),
				'default'     => '',
			),
			'merchant_key'     => array(
				'title'       => __( 'Merchant Key', 'woocommerce-gateway-smatpay' ),
				'type'        => 'text',
				'description' => __( '', 'woocommerce-gateway-smatpay' ),
				'default'     => '',
			),
            'merchant_api_key'      => array(
                'title'       => __( 'Merchant API Key', 'woocommerce-gateway-smatpay' ),
                'type'        => 'text',
                'description' => __( '', 'woocommerce-gateway-smatpay' ),
                'default'     => '',
            ),
            'payer_account_id'     => array(
                'title'       => __( 'Payer Account ID', 'woocommerce-gateway-smatpay' ),
                'type'        => 'text',
                'description' => __( '', 'woocommerce-gateway-smatpay' ),
                'default'     => '',
            ),
            'profile_id'     => array(
                'title'       => __( 'Profile ID', 'woocommerce-gateway-smatpay' ),
                'type'        => 'text',
                'description' => __( '', 'woocommerce-gateway-smatpay' ),
                'default'     => '',
            ),
//			'pass_phrase'      => array(
//				'title'       => __( 'Passphrase', 'woocommerce-gateway-smatpay' ),
//				'type'        => 'text',
//				'description' => __( '* Required. Needed to ensure the data passed through is secure.', 'woocommerce-gateway-smatpay' ),
//				'default'     => '',
//			),
//			'send_debug_email' => array(
//				'title'   => __( 'Send Debug Emails', 'woocommerce-gateway-smatpay' ),
//				'type'    => 'checkbox',
//				'label'   => __( 'Send debug e-mails for transactions through the Smatpay gateway (sends on successful transaction as well).', 'woocommerce-gateway-smatpay' ),
//				'default' => 'yes',
//			),
//			'debug_email'      => array(
//				'title'       => __( 'Who Receives Debug E-mails?', 'woocommerce-gateway-smatpay' ),
//				'type'        => 'text',
//				'description' => __( 'The e-mail address to which debugging error e-mails are sent when in test mode.', 'woocommerce-gateway-smatpay' ),
//				'default'     => get_option( 'admin_email' ),
//			),
//			'enable_logging'   => array(
//				'title'   => __( 'Enable Logging', 'woocommerce-gateway-smatpay' ),
//				'type'    => 'checkbox',
//				'label'   => __( 'Enable transaction logging for gateway.', 'woocommerce-gateway-smatpay' ),
//				'default' => 'no',
//			),
		);
	}

	/**
	 * Get the required form field keys for setup.
	 *
	 * @return array
	 */
	public function get_required_settings_keys() {
		return array(
			'merchant_id',
			'merchant_key',
            'merchant_api_key',
            'payer_account_id',
			'profile_id',
//            'pass_phrase'
		);
	}

	public function needs_setup() {
		return ! $this->get_option( 'merchant_id' ) ||
            ! $this->get_option( 'merchant_key' ) ||
            ! $this->get_option( 'merchant_api_key' ) ||
            ! $this->get_option( 'profile_id' ) ||
            ! $this->get_option( 'payer_account_id' );
	}

	public function add_testmode_admin_settings_notice() {
		$this->form_fields['merchant_api_key']['description']  .= ' <strong style="color: #bb0000;">' . esc_html__( 'Sandbox Merchant API Key currently in use', 'woocommerce-gateway-smatpay' ) . ' ( ' . esc_html( $this->merchant_api_key ) . ' ).</strong>';
	}

	/**
	 * Check if this gateway is enabled and available in the base currency being traded with.
	 *
	 * @since 1.0.0
	 * @return array
	 */
	public function check_requirements() {

		$errors = array(
			// Check if the store currency is supported by Smatpay.
			! in_array( get_woocommerce_currency(), $this->available_currencies, true ) ? 'wc-gateway-smatpay-error-invalid-currency' : null,
			// Check if user entered the merchant ID.
			'yes' !== $this->get_option( 'testmode' ) && empty( $this->get_option( 'merchant_id' ) ) ? 'wc-gateway-smatpay-error-missing-merchant-id' : null,
			// Check if user entered the merchant key.
			'yes' !== $this->get_option( 'testmode' ) && empty( $this->get_option( 'merchant_key' ) ) ? 'wc-gateway-smatpay-error-missing-merchant-key' : null,
		);

		return array_filter( $errors );
	}

	/**
	 * Check if the gateway is available for use.
	 *
	 * @return bool
	 */
	public function is_available() {
		if ( 'yes' === $this->enabled ) {
			$errors = $this->check_requirements();
			// Prevent using this gateway on frontend if there are any configuration errors.
			return 0 === count( $errors );
		}

		return parent::is_available();
	}

	/**
	 * Admin Panel Options
	 * - Options for bits like 'title' and availability on a country-by-country basis
	 *
	 * @since 1.0.0
	 */
	public function admin_options() {
		if ( in_array( get_woocommerce_currency(), $this->available_currencies, true ) ) {
			parent::admin_options();
		} else {
			?>
			<h3><?php esc_html_e( 'Smatpay', 'woocommerce-gateway-smatpay' ); ?></h3>
			<div class="inline error">
				<p>
					<strong><?php esc_html_e( 'Gateway Disabled', 'woocommerce-gateway-smatpay' ); ?></strong>
					<?php
					/* translators: 1: a href link 2: closing href */
					echo wp_kses_post( sprintf( __( 'Choose South African Rands as your store currency in %1$sGeneral Settings%2$s to enable the Smatpay Gateway.', 'woocommerce-gateway-smatpay' ), '<a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=general' ) ) . '">', '</a>' ) );
					?>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Generate the Smatpay button link.
	 *
	 * @since 1.0.0
	 *
	 * @param int $order_id Order ID.
	 * @return void
	 */
	public function generate_smatpay_form( $order_id ) {
		$order     = wc_get_order( $order_id );
		$site_name = html_entity_decode( get_bloginfo( 'name' ), ENT_QUOTES, get_bloginfo( 'charset' ) );
		// Construct variables for post.
		$this->data_to_send = array(
			// Merchant details.
			'merchant_id'      => $this->merchant_id,
			'merchant_key'     => $this->merchant_key,
            'testmode'         => $this->test_mode,
            'merchant_api_key' => $this->merchant_api_key,
            'payer_account_id' => $this->payer_account_id,
            'profile_id'       => $this->profile_id,
			'return_url'       => esc_url_raw( add_query_arg( 'utm_nooverride', '1', $this->get_return_url( $order ) ) ),
			'cancel_url'       => $order->get_cancel_order_url(),
			'notify_url'       => $this->response_url,

			// Billing details.
			'name_first'       => self::get_order_prop( $order, 'billing_first_name' ),
			'name_last'        => self::get_order_prop( $order, 'billing_last_name' ),
			'email_address'    => self::get_order_prop( $order, 'billing_email' ),

			// Item details.
			'm_payment_id'     => ltrim( $order->get_order_number(), _x( '#', 'hash before order number', 'woocommerce-gateway-smatpay' ) ),
			'amount'           => $order->get_total(),
            'order_details'    => $order,
            'order_key'        => $order->get_order_key(),
            'currency'         => get_woocommerce_currency(),
			'item_name'        => $site_name . ' - ' . $order->get_order_number(),
			/* translators: 1: blog info name */
			'item_description' => sprintf( esc_html__( 'New order from %s', 'woocommerce-gateway-smatpay' ), $site_name ),

			// Custom strings.
			'custom_str1'      => self::get_order_prop( $order, 'order_key' ),
			'custom_str2'      => 'WooCommerce/' . WC_VERSION . '; ' . rawurlencode( get_site_url() ),
			'custom_str3'      => self::get_order_prop( $order, 'id' ),
			'source'           => 'WooCommerce-Free-Plugin',
		);

		// Add Change subscription payment method parameters.
		if ( isset( $_GET['change_pay_method'] ) ) {
			$subscription_id = absint( wp_unslash( $_GET['change_pay_method'] ) );
			if ( $this->is_subscription( $subscription_id ) && $order_id === $subscription_id && floatval( 0 ) === floatval( $order->get_total() ) ) {
				$this->data_to_send['custom_str4'] = 'change_pay_method';
			}
		}

		/*
		 * Check If changing payment method.
		 * We have to generate Tokenization (ad-hoc) token to charge future payments.
		 */
		if ( $this->is_subscription( $order_id ) ) {
			// 2 == ad-hoc subscription type see Smatpay API docs
			$this->data_to_send['subscription_type'] = '2';
		}

		// Add subscription parameters.
		if ( $this->order_contains_subscription( $order_id ) ) {
			// 2 == ad-hoc subscription type see Smatpay API docs
			$this->data_to_send['subscription_type'] = '2';
		}

		if ( function_exists( 'wcs_order_contains_renewal' ) && wcs_order_contains_renewal( $order ) ) {
			$subscriptions = wcs_get_subscriptions_for_renewal_order( $order_id );
			$current       = reset( $subscriptions );
			// For renewal orders that have subscriptions with renewal flag OR
			// For renew orders which are failed to pay by other payment gateway buy now paying using Smatpay.
			// we will create a new subscription in Smatpay and link it to the existing ones in WC.
			// The old subscriptions in Smatpay will be cancelled once we handle the itn request.
			if ( count( $subscriptions ) > 0 && ( $this->_has_renewal_flag( $current ) || $this->id !== $current->get_payment_method() ) ) {
				// 2 == ad-hoc subscription type see Smatpay API docs
				$this->data_to_send['subscription_type'] = '2';
			}
		}

        // error_log(json_encode($this->data_to_send), 3, "errors/data_to_send_".time().".txt");

		/**
		 * Allow others to modify payment data before that is sent to Smatpay.
		 *
		 * @since 1.4.21
		 *
		 * @param array $this->data_to_send Payment data.
		 * @param int   $order_id           Order id.
		 */
		$this->data_to_send = apply_filters( 'woocommerce_gateway_smatpay_payment_data_to_send', $this->data_to_send, $order_id );

		$smatpay_args_array = array();
		$sign_strings       = array();
		foreach ( $this->data_to_send as $key => $value ) {
			if ( 'source' !== $key ) {
				// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.urlencode_urlencode -- legacy code, validation required prior to switching to rawurlencode.
				$sign_strings[] = esc_attr( $key ) . '=' . urlencode( str_replace( '&amp;', '&', trim( $value ) ) );
			}
			$smatpay_args_array[] = '<input type="hidden" name="' . esc_attr( $key ) . '" value="' . esc_attr( $value ) . '" />';
		}

//		if ( ! empty( $this->pass_phrase ) ) {
//			// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.urlencode_urlencode -- legacy code, validation required prior to switching to rawurlencode.
//			$smatpay_args_array[] = '<input type="hidden" name="signature" value="' . md5( implode( '&', $sign_strings ) . '&passphrase=' . urlencode( $this->pass_phrase ) ) . '" />';
//		} else {
//			$smatpay_args_array[] = '<input type="hidden" name="signature" value="' . md5( implode( '&', $sign_strings ) ) . '" />';
//		}

        // redirectUrl replaced this->url
        $redirectUrl = $this->generate_payment_link();

        header('Location: '.$redirectUrl);
        exit;

//		echo '<form action="' . esc_url( $redirectUrl ) . '" method="post" id="smatpay_payment_form">';
//
//		echo implode( '', $smatpay_args_array );
//		echo '
//				<input
//					type="submit"
//					class="button-alt"
//					id="submit_smatpay_payment_form"
//					value="' . esc_attr__( 'Pay via Smatpay', 'woocommerce-gateway-smatpay' ) . '"
//				/>
//				<a
//					class="button cancel"
//					href="' . esc_url( $order->get_cancel_order_url() ) . '"
//				>' .
//					esc_html__( 'Cancel order &amp; restore cart', 'woocommerce-gateway-smatpay' ) .
//				'</a>
//				<script type="text/javascript">
//					jQuery(function(){
//						// Feature detect.
//						if (
//							typeof PerformanceNavigationTiming !== "undefined" &&
//							typeof window.performance !== "undefined" &&
//							typeof performance.getEntriesByType === "function"
//						) {
//							var isBackForward = false;
//							var entries = performance.getEntriesByType("navigation");
//							entries.forEach((entry) => {
//								if (entry.type === "back_forward") {
//									isBackForward = true;
//								}
//							});
//							if (isBackForward) {
//								/*
//								 * Do not submit form on back or forward.
//								 * Ensure that the body is unblocked/not showing the redirect message.
//								 */
//								jQuery("body").unblock();
//								return;
//							}
//						}
//
//						jQuery("body").block(
//							{
//								message: "' . esc_html__( '<br> Thank you for your order. We are now redirecting you to Smatpay to make payment.', 'woocommerce-gateway-smatpay' ) . '",
//								overlayCSS:
//								{
//									background: "#fff",
//									opacity: 0.6
//								},
//								css: {
//									padding:        20,
//									textAlign:      "center",
//									color:          "#555",
//									border:         "3px solid #aaa",
//									backgroundColor:"#fff",
//									cursor:         "wait"
//								}
//							});
//						jQuery( "#submit_smatpay_payment_form" ).click();
//					});
//				</script>
//			</form>';
	}

    private function get_smatpay_currency()
    {
        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => SMATPAY_GET_CURRENCY_URL,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_SSL_VERIFYPEER => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'GET',
        ));

        $response = curl_exec($curl);

        curl_close($curl);

        if (curl_exec($curl) === false) {
            $error = curl_error($curl);
            throw new \Exception($error);
        }

        return $response;
    }

    /**
     * @throws Exception
     */
    private function generate_payment_link()
    {
        $currencies = json_decode($this->get_smatpay_currency(), true);

        $selected_currency = null;

        foreach($currencies as $currency) {
            if (strtolower($currency['currencyCode']) === strtolower($this->data_to_send['currency'])) {
                $selected_currency = $currency;
            }

            if (in_array( strtolower($this->data_to_send['currency']), ['zig', 'zwg']) && in_array(strtolower($currency['currencyCode']), ['zig', 'zwg'])) {
                $selected_currency = $currency;
            }
        }

        $payment_profile_id = intval($this->data_to_send['profile_id']);
        $amount = floatval($this->data_to_send['amount']);
        $payment_link_name = $this->data_to_send['item_name'];
        $payment_link_description = $this->data_to_send['item_description'];
        $payment_link_reference = $this->data_to_send['m_payment_id']."_". $this->data_to_send['custom_str1'];

        $query_params = [
            'page_id' => wc_get_page_id( 'checkout' ),
            'order-received' => $this->data_to_send['m_payment_id'],
            'key' => $this->data_to_send['custom_str1']
        ];

        $redirect_url = site_url() . '?' . http_build_query($query_params);

        $fastCheckoutBuilder = new FastCheckoutBuilder();

        $fastCheckoutBuilder
            ->setPaymentLinkName($payment_link_name)
            ->setPaymentLinkDescription($payment_link_description)
            ->setPaymentLinkReference($payment_link_reference)
            ->setPaymentLinkAmount($amount)
            ->setPaymentLinkType("ONCE_OFF_FIXED")
            ->setPaymentLinkImageType('NONE')
            ->setPaymentLinkProfileId($payment_profile_id)
            ->setPaymentLinkStartDate(date('Y-m-d'))
            ->setPaymentLinkEndDate(date('Y-m-d'))
            ->setPaymentLinkCurrency($selected_currency['currencyCode'])
            ->setPaymentLinkCurrencyId($selected_currency['id'])
            ->setPaymentLinkOtherCurrencies('false')
            ->setPaymentLinkCustomerRedirectUrl($redirect_url)
            ->setPaymentCustomerFailRedirectUrl($this->data_to_send['cancel_url'])
            ->setPaymentLinkEmail($this->data_to_send['email_address'])
            ->setPaymentPayerFullNames($this->data_to_send['name_first']." ".$this->data_to_send['name_last'])
            ->setPaymentPayerEmailAddress($this->data_to_send['email_address'])
            ->setPaymentPayerAddress("Address")
            ->setPaymentPayerMobile("+263772000000");

        $authBuilder = new AuthenticationBuilder();

        $authBuilder
            ->setMerchantKey($this->data_to_send['merchant_key'])
            ->setMerchantId($this->data_to_send['merchant_id'])
            ->setMerchantApiKey($this->data_to_send['merchant_api_key']);

        $fastCheckout = new FastCheckOut();
        $isSandbox = $this->data_to_send['testmode'] === "yes";

        $response = $fastCheckout->checkout($fastCheckoutBuilder, $authBuilder, $isSandbox);
        return $response->paymentLink->paymentLinkUrl;
    }

    /**
     * Generate the payment link redirect url
     *
     * @since 1.0.0
     *
     * @throws Exception When there is an error generating the token.
     * @
     *
     */
    private function generate_smatpay_token()
    {
        $request = [
            "merchantId" => $this->merchant_id,
            "merchantApiKey" => $this->merchant_api_key,
            "merchantKey" => $this->merchant_key
        ];

        $url = SANDBOX_SMATPAY_GET_TOKEN_URL;
        if (strtolower($this->test_mode) === 'no') {
            $url = PROD_SMATPAY_GET_TOKEN_URL;
        }

        $response = $this->api_call($url, $request);

        update_option('my_smatpay_token', $response->token);

        $savedToken = get_option('my_smatpay_token');
        error_log($savedToken.PHP_EOL.PHP_EOL, 3, 'smatpay_token_saved.txt');
        error_log($response->token.PHP_EOL.PHP_EOL, 3, 'smatpay_token_og.txt');
        error_log(json_encode($response).PHP_EOL.PHP_EOL, 3, 'smatpay_response.txt');
    }

    private function api_call($url, $payload_request, $has_auth = false, $is_json = true)
    {
        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_SSL_VERIFYPEER => 0,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => $is_json ? json_encode($payload_request) : $payload_request
        ));

        if ($is_json) {
            curl_setopt_array($curl, array(
                CURLOPT_HTTPHEADER => array(
                    'Content-Type: application/json'
                ),
            ));
        }
        if ($has_auth) {
            curl_setopt_array($curl, array(
                CURLOPT_HTTPHEADER => array(
                    'Authorization: Bearer '.get_option('my_smatpay_token')
                ),
            ));
        }

        $savedToken = get_option('my_smatpay_token');
        error_log($savedToken.PHP_EOL.PHP_EOL, 3, 'smatpay_token_saved_2.txt');

		echo $url.PHP_EOL;

		echo json_encode($payload_request).PHP_EOL;

        die();

        $response = curl_exec($curl);

		echo $response;

		echo "This code was activated!!!!!!!";

        curl_close($curl);

        if (curl_exec($curl) === false) {
            throw new \Exception("Failed to generate redirect url!");
        }

        $result = json_decode($response);
        if (isset($result->error)) {
            throw new \Exception($result->error);
        }

        return $result;
    }

	/**
	 * Process the payment and return the result.
	 *
	 * @since 1.0.0
	 *
	 * @throws Exception When there is an error processing the payment.
	 *
	 * @param int $order_id Order ID.
	 * @return string[] Payment result {
	 *    @type string $result   Result of payment.
	 *    @type string $redirect Redirect URL.
	 * }
	 */
	public function process_payment( $order_id ) {
		$order    = wc_get_order( $order_id );
		$redirect = $order->get_checkout_payment_url( true );

		// Check if the payment is for changing payment method.
		if ( isset( $_GET['change_payment_method'] ) ) {
			$sub_id = absint( wp_unslash( $_GET['change_payment_method'] ) );
			if ( $this->is_subscription( $sub_id ) && floatval( 0 ) === floatval( $order->get_total() ) ) {
				$redirect = add_query_arg( 'change_pay_method', $sub_id, $redirect );
			}
		}

        error_log($this->get_return_url($order), 3, 'redirect_url.txt');

        $order->reduce_order_stock();
        WC()->cart->empty_cart();

		return array(
			'result'   => 'success',
			'redirect' => $redirect,
		);
	}

	/**
	 * Reciept page.
	 *
	 * Display text and a button to direct the user to Smatpay.
	 *
	 * @param WC_Order $order Order object.
	 * @since 1.0.0
	 */
	public function receipt_page( $order ) {
		echo '<p>' . esc_html__( 'Thank you for your order, please click the button below to pay with Smatpay.', 'woocommerce-gateway-smatpay' ) . '</p>';
		$this->generate_smatpay_form( $order );
	}

	/**
	 * Check Smatpay ITN response.
	 *
	 * @since 1.0.0
	 */
	public function check_itn_response() {
		// phpcs:ignore.WordPress.Security.NonceVerification.Missing
		$this->handle_itn_request( stripslashes_deep( $_POST ) );

		// Notify Smatpay that information has been received.
		header( 'HTTP/1.0 200 OK' );
		flush();
	}

	/**
	 * Check Smatpay ITN validity.
	 *
	 * @param array $data Data.
	 * @since 1.0.0
	 */
	public function handle_itn_request( $data ) {
		$this->log(
			PHP_EOL
			. '----------'
			. PHP_EOL . 'Smatpay ITN call received'
			. PHP_EOL . '----------'
		);
		$this->log( 'Get posted data' );
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r -- debug info for logging.
		$this->log( 'Smatpay Data: ' . print_r( $data, true ) );

		$smatpay_error  = false;
		$smatpay_done   = false;
		$debug_email    = $this->get_option( 'debug_email', get_option( 'admin_email' ) );
		$session_id     = $data['custom_str1'];
		$vendor_name    = get_bloginfo( 'name', 'display' );
		$vendor_url     = home_url( '/' );
		$order_id       = absint( $data['custom_str3'] );
		$order_key      = wc_clean( $session_id );
		$order          = wc_get_order( $order_id );
		$original_order = $order;

		if ( false === $data ) {
			$smatpay_error         = true;
			$smatpay_error_message = PF_ERR_BAD_ACCESS;
		}

		// Verify security signature.
		if ( ! $smatpay_error && ! $smatpay_done ) {
			$this->log( 'Verify security signature' );
			$signature = md5( $this->_generate_parameter_string( $data, false, false ) ); // false not to sort data.
			// If signature different, log for debugging.
			if ( ! $this->validate_signature( $data, $signature ) ) {
				$smatpay_error         = true;
				$smatpay_error_message = PF_ERR_INVALID_SIGNATURE;
			}
		}

		// Verify source IP (If not in debug mode).
		if ( ! $smatpay_error && ! $smatpay_done
			&& $this->get_option( 'testmode' ) !== 'yes' ) {
			$this->log( 'Verify source IP' );

			if ( isset( $_SERVER['REMOTE_ADDR'] ) && ! $this->is_valid_ip( sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) ) ) {
				$smatpay_error         = true;
				$smatpay_error_message = PF_ERR_BAD_SOURCE_IP;
			}
		}

		// Verify data received.
		if ( ! $smatpay_error ) {
			$this->log( 'Verify data received' );
			$validation_data = $data;
			unset( $validation_data['signature'] );
			$has_valid_response_data = $this->validate_response_data( $validation_data );

			if ( ! $has_valid_response_data ) {
				$smatpay_error         = true;
				$smatpay_error_message = PF_ERR_BAD_ACCESS;
			}
		}

		/**
		 * Handle Changing Payment Method.
		 *   - Save smatpay subscription token to handle future payment
		 *   - (for Smatpay to Smatpay payment method change) Cancel old token, as future payment will be handle with new token
		 *
		 * Note: The change payment method is handled before the amount mismatch check, as it doesn't involve an actual payment (0.00) and only token updates are handled here.
		 */
		if (
			! $smatpay_error &&
			isset( $data['custom_str4'] ) &&
			'change_pay_method' === wc_clean( $data['custom_str4'] ) &&
			$this->is_subscription( $order_id ) &&
			floatval( 0 ) === floatval( $data['amount_gross'] )
		) {
			if ( self::get_order_prop( $order, 'order_key' ) !== $order_key ) {
				$this->log( 'Order key does not match' );
				exit;
			}

			$this->log( '- Change Payment Method' );
			$status = strtolower( $data['payment_status'] );
			if ( 'complete' === $status && isset( $data['token'] ) ) {
				$token        = sanitize_text_field( $data['token'] );
				$subscription = wcs_get_subscription( $order_id );
				if ( ! empty( $subscription ) && ! empty( $token ) ) {
					$old_token = $this->_get_subscription_token( $subscription );
					// Cancel old subscription token of subscription if we have it.
					if ( ! empty( $old_token ) ) {
						$this->cancel_subscription_listener( $subscription );
					}

					// Set new subscription token on subscription.
					$this->_set_subscription_token( $token, $subscription );
					$this->log( 'Smatpay token updated on Subcription: ' . $order_id );
				}
			}
			return;
		}

		// Check data against internal order.
		if ( ! $smatpay_error && ! $smatpay_done ) {
			$this->log( 'Check data against internal order' );

			// alter order object to be the renewal order if
			// the ITN request comes as a result of a renewal submission request.
			$description = json_decode( $data['item_description'] );

			if ( ! empty( $description->renewal_order_id ) ) {
				$renewal_order = wc_get_order( $description->renewal_order_id );
				if ( ! empty( $renewal_order ) && function_exists( 'wcs_order_contains_renewal' ) && wcs_order_contains_renewal( $renewal_order ) ) {
					$order = $renewal_order;
				}
			}

			// Check order amount.
			if ( ! $this->amounts_equal( $data['amount_gross'], self::get_order_prop( $order, 'order_total' ) ) ) {
				$smatpay_error         = true;
				$smatpay_error_message = PF_ERR_AMOUNT_MISMATCH;
			} elseif ( strcasecmp( $data['custom_str1'], self::get_order_prop( $original_order, 'order_key' ) ) !== 0 ) {
				// Check session ID.
				$smatpay_error         = true;
				$smatpay_error_message = PF_ERR_SESSIONID_MISMATCH;
			}
		}

		// Get internal order and verify it hasn't already been processed.
		if ( ! $smatpay_error && ! $smatpay_done ) {
			$this->log_order_details( $order );

			// Check if order has already been processed.
			if ( 'completed' === self::get_order_prop( $order, 'status' ) ) {
				$this->log( 'Order has already been processed' );
				$smatpay_done = true;
			}
		}

		// If an error occurred.
		if ( $smatpay_error ) {
			$this->log( 'Error occurred: ' . $smatpay_error_message );

			if ( $this->send_debug_email ) {
				$this->log( 'Sending email notification' );

				// Send an email.
				$subject = 'Smatpay ITN error: ' . $smatpay_error_message;
				$body    =
					"Hi,\n\n" .
					"An invalid Smatpay transaction on your website requires attention\n" .
					"------------------------------------------------------------\n" .
					'Site: ' . esc_html( $vendor_name ) . ' (' . esc_url( $vendor_url ) . ")\n" .
					'Remote IP Address: ' . sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) . "\n" .
					'Remote host name: ' . gethostbyaddr( sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) ) . "\n" .
					'Purchase ID: ' . self::get_order_prop( $order, 'id' ) . "\n" .
					'User ID: ' . self::get_order_prop( $order, 'user_id' ) . "\n";
				if ( isset( $data['pf_payment_id'] ) ) {
					$body .= 'Smatpay Transaction ID: ' . esc_html( $data['pf_payment_id'] ) . "\n";
				}
				if ( isset( $data['payment_status'] ) ) {
					$body .= 'Smatpay Payment Status: ' . esc_html( $data['payment_status'] ) . "\n";
				}

				$body .= "\nError: " . $smatpay_error_message . "\n";

				switch ( $smatpay_error_message ) {
					case PF_ERR_AMOUNT_MISMATCH:
						$body .=
							'Value received : ' . esc_html( $data['amount_gross'] ) . "\n"
							. 'Value should be: ' . self::get_order_prop( $order, 'order_total' );
						break;

					case PF_ERR_ORDER_ID_MISMATCH:
						$body .=
							'Value received : ' . esc_html( $data['custom_str3'] ) . "\n"
							. 'Value should be: ' . self::get_order_prop( $order, 'id' );
						break;

					case PF_ERR_SESSIONID_MISMATCH:
						$body .=
							'Value received : ' . esc_html( $data['custom_str1'] ) . "\n"
							. 'Value should be: ' . self::get_order_prop( $order, 'id' );
						break;

					// For all other errors there is no need to add additional information.
					default:
						break;
				}

				wp_mail( $debug_email, $subject, $body );
			} // End if.
		} elseif ( ! $smatpay_done ) {

			$this->log( 'Check status and update order' );

			if ( self::get_order_prop( $original_order, 'order_key' ) !== $order_key ) {
				$this->log( 'Order key does not match' );
				exit;
			}

			$status = strtolower( $data['payment_status'] );

			$subscriptions = array();
			if ( function_exists( 'wcs_get_subscriptions_for_renewal_order' ) && function_exists( 'wcs_get_subscriptions_for_order' ) ) {
				$subscriptions = array_merge(
					wcs_get_subscriptions_for_renewal_order( $order_id ),
					wcs_get_subscriptions_for_order( $order_id )
				);
			}

			if ( 'complete' !== $status && 'cancelled' !== $status ) {
				foreach ( $subscriptions as $subscription ) {
					$this->_set_renewal_flag( $subscription );
				}
			}

			if ( 'complete' === $status ) {
				$this->handle_itn_payment_complete( $data, $order, $subscriptions );
			} elseif ( 'failed' === $status ) {
				$this->handle_itn_payment_failed( $data, $order );
			} elseif ( 'pending' === $status ) {
				$this->handle_itn_payment_pending( $data, $order );
			} elseif ( 'cancelled' === $status ) {
				$this->handle_itn_payment_cancelled( $data, $order, $subscriptions );
			}
		} // End if.

		$this->log(
			PHP_EOL
			. '----------'
			. PHP_EOL . 'End ITN call'
			. PHP_EOL . '----------'
		);
	}

	/**
	 * Handle logging the order details.
	 *
	 * @since 1.4.5
	 *
	 * @param WC_Order $order Order object.
	 */
	public function log_order_details( $order ) {
		$customer_id = $order->get_user_id();

		$details = 'Order Details:'
		. PHP_EOL . 'customer id:' . $customer_id
		. PHP_EOL . 'order id:   ' . $order->get_id()
		. PHP_EOL . 'parent id:  ' . $order->get_parent_id()
		. PHP_EOL . 'status:     ' . $order->get_status()
		. PHP_EOL . 'total:      ' . $order->get_total()
		. PHP_EOL . 'currency:   ' . $order->get_currency()
		. PHP_EOL . 'key:        ' . $order->get_order_key()
		. '';

		$this->log( $details );
	}

	/**
	 * This function mainly responds to ITN cancel requests initiated on Smatpay, but also acts
	 * just in case they are not cancelled.
	 *
	 * @version 1.4.3 Subscriptions flag
	 *
	 * @param array             $data          Should be from the Gateway ITN callback.
	 * @param WC_Order          $order         Order object.
	 * @param WC_Subscription[] $subscriptions Array of subscriptions.
	 */
	public function handle_itn_payment_cancelled( $data, $order, $subscriptions ) {

		remove_action( 'woocommerce_subscription_status_cancelled', array( $this, 'cancel_subscription_listener' ) );
		foreach ( $subscriptions as $subscription ) {
			if ( 'cancelled' !== $subscription->get_status() ) {
				$subscription->update_status( 'cancelled', esc_html__( 'Merchant cancelled subscription on Smatpay.', 'woocommerce-gateway-smatpay' ) );
				$this->_delete_subscription_token( $subscription );
			}
		}
		add_action( 'woocommerce_subscription_status_cancelled', array( $this, 'cancel_subscription_listener' ) );
	}

	/**
	 * This function handles payment complete request by Smatpay.
	 *
	 * @version 1.4.3 Subscriptions flag
	 *
	 * @param array             $data          Should be from the Gateway ITN callback.
	 * @param WC_Order          $order         Order object.
	 * @param WC_Subscription[] $subscriptions Array of subscriptions.
	 */
	public function handle_itn_payment_complete( $data, $order, $subscriptions ) {
		$this->log( '- Complete' );
		$order->add_order_note( esc_html__( 'ITN payment completed', 'woocommerce-gateway-smatpay' ) );
		$order->update_meta_data( 'smatpay_amount_fee', $data['amount_fee'] );
		$order->update_meta_data( 'smatpay_amount_net', $data['amount_net'] );
		$order_id = self::get_order_prop( $order, 'id' );

		// Store token for future subscription deductions.
		if ( count( $subscriptions ) > 0 && isset( $data['token'] ) ) {
			if ( $this->_has_renewal_flag( reset( $subscriptions ) ) ) {
				// Renewal flag is set to true, so we need to cancel previous token since we will create a new one.
				$this->log( 'Cancel previous subscriptions with token ' . $this->_get_subscription_token( reset( $subscriptions ) ) );

				// Only request API cancel token for the first subscription since all of them are using the same token.
				$this->cancel_subscription_listener( reset( $subscriptions ) );
			}

			$token = sanitize_text_field( $data['token'] );
			foreach ( $subscriptions as $subscription ) {
				$this->_delete_renewal_flag( $subscription );
				$this->_set_subscription_token( $token, $subscription );
			}
		}

		// Mark payment as complete.
		$order->payment_complete( $data['pf_payment_id'] );

		$debug_email = $this->get_option( 'debug_email', get_option( 'admin_email' ) );
		$vendor_name = get_bloginfo( 'name', 'display' );
		$vendor_url  = home_url( '/' );
		if ( $this->send_debug_email ) {
			$subject = 'Smatpay ITN on your site';
			$body    =
				"Hi,\n\n"
				. "A Smatpay transaction has been completed on your website\n"
				. "------------------------------------------------------------\n"
				. 'Site: ' . esc_html( $vendor_name ) . ' (' . esc_url( $vendor_url ) . ")\n"
				. 'Purchase ID: ' . esc_html( $data['m_payment_id'] ) . "\n"
				. 'Smatpay Transaction ID: ' . esc_html( $data['pf_payment_id'] ) . "\n"
				. 'Smatpay Payment Status: ' . esc_html( $data['payment_status'] ) . "\n"
				. 'Order Status Code: ' . self::get_order_prop( $order, 'status' );
			wp_mail( $debug_email, $subject, $body );
		}

		/**
		 * Fires after handling the Payment Complete ITN from Smatpay.
		 *
		 * @since 1.4.22
		 *
		 * @param array             $data          ITN Payload.
		 * @param WC_Order          $order         Order Object.
		 * @param WC_Subscription[] $subscriptions Subscription array.
		 */
		do_action( 'woocommerce_smatpay_handle_itn_payment_complete', $data, $order, $subscriptions );
	}

	/**
	 * Handle payment failed request by Smatpay.
	 *
	 * @param array    $data  Should be from the Gateway ITN callback.
	 * @param WC_Order $order Order object.
	 */
	public function handle_itn_payment_failed( $data, $order ) {
		$this->log( '- Failed' );
		/* translators: 1: payment status */
		$order->update_status( 'failed', sprintf( __( 'Payment %s via ITN.', 'woocommerce-gateway-smatpay' ), strtolower( sanitize_text_field( $data['payment_status'] ) ) ) );
		$debug_email = $this->get_option( 'debug_email', get_option( 'admin_email' ) );
		$vendor_name = get_bloginfo( 'name', 'display' );
		$vendor_url  = home_url( '/' );

		if ( $this->send_debug_email ) {
			$subject = 'Smatpay ITN Transaction on your site';
			$body    =
				"Hi,\n\n" .
				"A failed Smatpay transaction on your website requires attention\n" .
				"------------------------------------------------------------\n" .
				'Site: ' . esc_html( $vendor_name ) . ' (' . esc_url( $vendor_url ) . ")\n" .
				'Purchase ID: ' . self::get_order_prop( $order, 'id' ) . "\n" .
				'User ID: ' . self::get_order_prop( $order, 'user_id' ) . "\n" .
				'Smatpay Transaction ID: ' . esc_html( $data['pf_payment_id'] ) . "\n" .
				'Smatpay Payment Status: ' . esc_html( $data['payment_status'] );
			wp_mail( $debug_email, $subject, $body );
		}
	}

	/**
	 * Handle payment pending request by Smatpay.
	 *
	 * @since 1.4.0
	 *
	 * @param array    $data  Should be from the Gateway ITN callback.
	 * @param WC_Order $order Order object.
	 */
	public function handle_itn_payment_pending( $data, $order ) {
		$this->log( '- Pending' );
		// Need to wait for "Completed" before processing.
		/* translators: 1: payment status */
		$order->update_status( 'on-hold', sprintf( esc_html__( 'Payment %s via ITN.', 'woocommerce-gateway-smatpay' ), strtolower( sanitize_text_field( $data['payment_status'] ) ) ) );
	}

	/**
	 * Get the pre-order fee.
	 *
	 * @param string $order_id Order ID.
	 * @return double
	 */
	public function get_pre_order_fee( $order_id ) {
		foreach ( wc_get_order( $order_id )->get_fees() as $fee ) {
			if ( is_array( $fee ) && 'Pre-Order Fee' === $fee['name'] ) {
				return doubleval( $fee['line_total'] ) + doubleval( $fee['line_tax'] );
			}
		}
	}

	/**
	 * Whether order contains a pre-order.
	 *
	 * @param string $order_id Order ID.
	 * @return bool Whether order contains a pre-order.
	 */
	public function order_contains_pre_order( $order_id ) {
		if ( class_exists( 'WC_Pre_Orders_Order' ) ) {
			return WC_Pre_Orders_Order::order_contains_pre_order( $order_id );
		}
		return false;
	}

	/**
	 * Whether the order requires payment tokenization.
	 *
	 * @param string $order_id Order ID.
	 * @return bool Whether the order requires payment tokenization.
	 */
	public function order_requires_payment_tokenization( $order_id ) {
		if ( class_exists( 'WC_Pre_Orders_Order' ) ) {
			return WC_Pre_Orders_Order::order_requires_payment_tokenization( $order_id );
		}
		return false;
	}

	/**
	 * Whether order contains a pre-order fee.
	 *
	 * @return bool Whether order contains a pre-order fee.
	 */
	public function cart_contains_pre_order_fee() {
		if ( class_exists( 'WC_Pre_Orders_Cart' ) ) {
			return WC_Pre_Orders_Cart::cart_contains_pre_order_fee();
		}
		return false;
	}
	/**
	 * Store the Smatpay subscription token
	 *
	 * @param string          $token        Smatpay subscription token.
	 * @param WC_Subscription $subscription The subscription object.
	 */
	protected function _set_subscription_token( $token, $subscription ) {
		$subscription->update_meta_data( '_smatpay_subscription_token', $token );
		$subscription->save_meta_data();
	}

	/**
	 * Retrieve the Smatpay subscription token for a given order id.
	 *
	 * @param WC_Subscription $subscription The subscription object.
	 * @return mixed Smatpay subscription token.
	 */
	protected function _get_subscription_token( $subscription ) {
		return $subscription->get_meta( '_smatpay_subscription_token', true );
	}

	/**
	 * Retrieve the Smatpay subscription token for a given order id.
	 *
	 * @param WC_Subscription $subscription The subscription object.
	 * @return mixed
	 */
	protected function _delete_subscription_token( $subscription ) {
		return $subscription->delete_meta_data( '_smatpay_subscription_token' );
	}

	/**
	 * Store the Smatpay renewal flag
	 *
	 * @since 1.4.3
	 *
	 * @param WC_Subscription $subscription The subscription object.
	 */
	protected function _set_renewal_flag( $subscription ) {
		$subscription->update_meta_data( '_smatpay_renewal_flag', 'true' );
		$subscription->save_meta_data();
	}

	/**
	 * Retrieve the Smatpay renewal flag for a given order id.
	 *
	 * @since 1.4.3
	 *
	 * @param WC_Subscription $subscription The subscription object.
	 * @return bool
	 */
	protected function _has_renewal_flag( $subscription ) {
		return 'true' === $subscription->get_meta( '_smatpay_renewal_flag', true );
	}

	/**
	 * Retrieve the Smatpay renewal flag for a given order id.
	 *
	 * @since 1.4.3
	 *
	 * @param WC_Subscription $subscription The subscription object.
	 */
	protected function _delete_renewal_flag( $subscription ) {
		$subscription->delete_meta_data( '_smatpay_renewal_flag' );
		$subscription->save_meta_data();
	}

	/**
	 * Wrapper for WooCommerce subscription function wc_is_subscription.
	 *
	 * @param WC_Order|int $order The order.
	 * @return bool
	 */
	public function is_subscription( $order ) {
		if ( ! function_exists( 'wcs_is_subscription' ) ) {
			return false;
		}
		return wcs_is_subscription( $order );
	}

	/**
	 * Cancel Smatpay Tokenization(ad-hoc) token if subscription changed to other payment method.
	 *
	 * @param WC_Subscription $subscription       The subscription for which the payment method changed.
	 * @param string          $new_payment_method New payment method name.
	 */
	public function maybe_cancel_subscription_token( $subscription, $new_payment_method ) {
		$token = $this->_get_subscription_token( $subscription );
		if ( empty( $token ) || $this->id === $new_payment_method ) {
			return;
		}
		$this->cancel_subscription_listener( $subscription );
		$this->_delete_subscription_token( $subscription );

		$this->log( 'Smatpay subscription token Cancelled.' );
	}

	/**
	 * Whether order contains a subscription.
	 *
	 * Wrapper function for wcs_order_contains_subscription
	 *
	 * @param WC_Order $order Order object.
	 * @return bool Whether order contains a subscription.
	 */
	public function order_contains_subscription( $order ) {
		if ( ! function_exists( 'wcs_order_contains_subscription' ) ) {
			return false;
		}
		return wcs_order_contains_subscription( $order );
	}

	/**
	 * Process scheduled subscription payment and update the subscription status accordingly.
	 *
	 * @param float    $amount_to_charge Subscription cost.
	 * @param WC_Order $renewal_order    Renewal order object.
	 */
	public function scheduled_subscription_payment( $amount_to_charge, $renewal_order ) {

		$subscription = wcs_get_subscription( $renewal_order->get_meta( '_subscription_renewal', true ) );
		$this->log( 'Attempting to renew subscription from renewal order ' . self::get_order_prop( $renewal_order, 'id' ) );

		if ( empty( $subscription ) ) {
			$this->log( 'Subscription from renewal order was not found.' );
			return;
		}

		$response = $this->submit_subscription_payment( $subscription, $amount_to_charge );

		if ( is_wp_error( $response ) ) {
			/* translators: 1: error code 2: error message */
			$renewal_order->update_status( 'failed', sprintf( esc_html__( 'Smatpay Subscription renewal transaction failed (%1$s:%2$s)', 'woocommerce-gateway-smatpay' ), $response->get_error_code(), $response->get_error_message() ) );
		}
		// Payment will be completion will be capture only when the ITN callback is sent to $this->handle_itn_request().
		$renewal_order->add_order_note( esc_html__( 'Smatpay Subscription renewal transaction submitted.', 'woocommerce-gateway-smatpay' ) );
	}

	/**
	 * Attempt to process a subscription payment on the Smatpay gateway.
	 *
	 * @param WC_Subscription $subscription     The subscription object.
	 * @param float           $amount_to_charge The amount to charge.
	 * @return mixed WP_Error on failure, bool true on success
	 */
	public function submit_subscription_payment( $subscription, $amount_to_charge ) {
		$token     = $this->_get_subscription_token( $subscription );
		$item_name = $this->get_subscription_name( $subscription );

		foreach ( $subscription->get_related_orders( 'all', 'renewal' ) as $order ) {
			$statuses_to_charge = array( 'on-hold', 'failed', 'pending' );
			if ( in_array( $order->get_status(), $statuses_to_charge, true ) ) {
				$latest_order_to_renew = $order;
				break;
			}
		}
		$item_description = wp_json_encode( array( 'renewal_order_id' => self::get_order_prop( $latest_order_to_renew, 'id' ) ) );

		return $this->submit_ad_hoc_payment( $token, $amount_to_charge, $item_name, $item_description );
	}

	/**
	 * Get a name for the subscription item. For multiple
	 * item only Subscription $date will be returned.
	 *
	 * For subscriptions with no items Site/Blog name will be returned.
	 *
	 * @param WC_Subscription $subscription The subscription object.
	 * @return string
	 */
	public function get_subscription_name( $subscription ) {

		if ( $subscription->get_item_count() > 1 ) {
			return $subscription->get_date_to_display( 'start' );
		} else {
			$items = $subscription->get_items();

			if ( empty( $items ) ) {
				return get_bloginfo( 'name' );
			}

			$item = array_shift( $items );
			return $item['name'];
		}
	}

	/**
	 * Setup api data for the the adhoc payment.
	 *
	 * @since 1.4.0 introduced.
	 *
	 * @param string $token            Smatpay subscription token.
	 * @param float  $amount_to_charge Amount to charge.
	 * @param string $item_name        Item name.
	 * @param string $item_description Item description.
	 *
	 * @return bool|WP_Error WP_Error on failure, bool true on success
	 */
	public function submit_ad_hoc_payment( $token, $amount_to_charge, $item_name, $item_description ) {
		$args = array(
			'body' => array(
				'amount'           => $amount_to_charge * 100, // Convert to cents.
				'item_name'        => $item_name,
				'item_description' => $item_description,
			),
		);
		return $this->api_request( 'adhoc', $token, $args );
	}

	/**
	 * Send off API request.
	 *
	 * @since 1.4.0 introduced.
	 *
	 * @param string $command  API command.
	 * @param string $token    Smatpay subscription token.
	 * @param array  $api_args Arguments for the API request. See WP documentation for wp_remote_request.
	 * @param string $method   GET | PUT | POST | DELETE.
	 *
	 * @return bool|WP_Error WP_Error on failure, bool true on success
	 */
	public function api_request( $command, $token, $api_args, $method = 'POST' ) {
		if ( empty( $token ) ) {
			$this->log( 'Error posting API request: No token supplied', true );
			return new WP_Error( '404', esc_html__( 'Can not submit Smatpay request with an empty token', 'woocommerce-gateway-smatpay' ), $results );
		}

		$api_endpoint  = "https://api.smatpay.co.za/subscriptions/$token/$command";
		$api_endpoint .= 'yes' === $this->get_option( 'testmode' ) ? '?testing=true' : '';

		$timestamp           = current_time( rtrim( DateTime::ATOM, 'P' ) ) . '+02:00';
		$api_args['timeout'] = 45;
		$api_args['headers'] = array(
			'merchant-id' => $this->merchant_id,
			'timestamp'   => $timestamp,
			'version'     => 'v1',
		);

		// Set content length to fix "411: requests require a Content-length header" error.
		if ( 'cancel' === $command && ! isset( $api_args['body'] ) ) {
			$api_args['headers']['content-length'] = 0;
		}

		// Generate signature.
		$all_api_variables                = array_merge( $api_args['headers'], (array) $api_args['body'] );
		$api_args['headers']['signature'] = md5( $this->_generate_parameter_string( $all_api_variables ) );
		$api_args['method']               = strtoupper( $method );

		$results = wp_remote_request( $api_endpoint, $api_args );

		if ( is_wp_error( $results ) ) {
			return $results;
		}

		// Check Smatpay server response.
		if ( 200 !== $results['response']['code'] ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r -- used for logging.
			$this->log( "Error posting API request:\n" . print_r( $results['response'], true ) );
			return new WP_Error( $results['response']['code'], json_decode( $results['body'] )->data->response, $results );
		}

		// Check adhoc bank charge response.
		$results_data = json_decode( $results['body'], true )['data'];

		// Sandbox ENV returns true(boolean) in response, while Production ENV "true"(string) in response.
		if ( 'adhoc' === $command && ! ( 'true' === $results_data['response'] || true === $results_data['response'] ) ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r -- used for logging.
			$this->log( "Error posting API request:\n" . print_r( $results_data, true ) );

			$code    = is_array( $results_data['response'] ) ? $results_data['response']['code'] : $results_data['response'];
			$message = is_array( $results_data['response'] ) ? $results_data['response']['reason'] : $results_data['message'];
			// Use trim here to display it properly e.g. on an order note, since Smatpay can include CRLF in a message.
			return new WP_Error( $code, trim( $message ), $results );
		}

		$maybe_json = json_decode( $results['body'], true );

		if ( ! is_null( $maybe_json ) && isset( $maybe_json['status'] ) && 'failed' === $maybe_json['status'] ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r -- used for logging.
			$this->log( "Error posting API request:\n" . print_r( $results['body'], true ) );

			// Use trim here to display it properly e.g. on an order note, since Smatpay can include CRLF in a message.
			return new WP_Error( $maybe_json['code'], trim( $maybe_json['data']['message'] ), $results['body'] );
		}

		return true;
	}

	/**
	 * Responds to Subscriptions extension cancellation event.
	 *
	 * @since 1.4.0 introduced.
	 * @param WC_Subscription $subscription The subscription object.
	 */
	public function cancel_subscription_listener( $subscription ) {
		$token = $this->_get_subscription_token( $subscription );
		if ( empty( $token ) ) {
			return;
		}
		$this->api_request( 'cancel', $token, array(), 'PUT' );
	}

	/**
	 * Cancel a pre-order subscription.
	 *
	 * @since 1.4.0
	 *
	 * @param string $token Smatpay subscription token.
	 *
	 * @return bool|WP_Error WP_Error on failure, bool true on success.
	 */
	public function cancel_pre_order_subscription( $token ) {
		return $this->api_request( 'cancel', $token, array(), 'PUT' );
	}

	/**
	 * Generate the parameter string to send to Smatpay.
	 *
	 * @since 1.4.0 introduced.
	 *
	 * @param array $api_data               Data to send to the Smatpay API.
	 * @param bool  $sort_data_before_merge Whether to sort before merge. Default true.
	 * @param bool  $skip_empty_values      Should key value pairs be ignored when generating signature? Default true.
	 *
	 * @return string
	 */
	protected function _generate_parameter_string( $api_data, $sort_data_before_merge = true, $skip_empty_values = true ) {

		// if sorting is required the passphrase should be added in before sort.
		if ( ! empty( $this->pass_phrase ) && $sort_data_before_merge ) {
			$api_data['passphrase'] = $this->pass_phrase;
		}

		if ( $sort_data_before_merge ) {
			ksort( $api_data );
		}

		// concatenate the array key value pairs.
		$parameter_string = '';
		foreach ( $api_data as $key => $val ) {

			if ( $skip_empty_values && empty( $val ) ) {
				continue;
			}

			if ( 'signature' !== $key ) {
				// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.urlencode_urlencode -- legacy code, validation required prior to switching to rawurlencode.
				$val               = urlencode( $val );
				$parameter_string .= "$key=$val&";
			}
		}
		// When not sorting passphrase should be added to the end before md5.
		if ( $sort_data_before_merge ) {
			$parameter_string = rtrim( $parameter_string, '&' );
		} elseif ( ! empty( $this->pass_phrase ) ) {
			// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.urlencode_urlencode -- legacy code, validation required prior to switching to rawurlencode.
			$parameter_string .= 'passphrase=' . urlencode( $this->pass_phrase );
		} else {
			$parameter_string = rtrim( $parameter_string, '&' );
		}

		return $parameter_string;
	}

	/**
	 * Process pre-order payment.
	 *
	 * @since 1.4.0 introduced.
	 *
	 * @param WC_Order $order Order object.
	 */
	public function process_pre_order_payments( $order ) {
		wc_deprecated_function( 'process_pre_order_payments', '1.6.3' );
	}

	/**
	 * Setup constants.
	 *
	 * Setup common values and messages used by the Smatpay gateway.
	 *
	 * @since 1.0.0
	 */
	public function setup_constants() {
		// Create user agent string.
		define( 'PF_SOFTWARE_NAME', 'WooCommerce' );
		define( 'PF_SOFTWARE_VER', WC_VERSION );
		define( 'PF_MODULE_NAME', 'WooCommerce-Smatpay-Free' );
		define( 'PF_MODULE_VER', $this->version );

		// Features
		// - PHP.
		$pf_features = 'PHP ' . phpversion() . ';';

		// - cURL.
		if ( in_array( 'curl', get_loaded_extensions(), true ) ) {
			define( 'PF_CURL', '' );
			$pf_version   = curl_version();
			$pf_features .= ' curl ' . $pf_version['version'] . ';';
		} else {
			$pf_features .= ' nocurl;';
		}

		// Create user agent.
		define( 'PF_USER_AGENT', PF_SOFTWARE_NAME . '/' . PF_SOFTWARE_VER . ' (' . trim( $pf_features ) . ') ' . PF_MODULE_NAME . '/' . PF_MODULE_VER );

		// General Defines.
		define( 'PF_TIMEOUT', 15 );
		define( 'PF_EPSILON', 0.01 );

		// Error messages.
		define( 'PF_ERR_AMOUNT_MISMATCH', esc_html__( 'Amount mismatch', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_BAD_ACCESS', esc_html__( 'Bad access of page', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_BAD_SOURCE_IP', esc_html__( 'Bad source IP address', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_CONNECT_FAILED', esc_html__( 'Failed to connect to Smatpay', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_INVALID_SIGNATURE', esc_html__( 'Security signature mismatch', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_MERCHANT_ID_MISMATCH', esc_html__( 'Merchant ID mismatch', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_NO_SESSION', esc_html__( 'No saved session found for ITN transaction', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_ORDER_ID_MISSING_URL', esc_html__( 'Order ID not present in URL', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_ORDER_ID_MISMATCH', esc_html__( 'Order ID mismatch', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_ORDER_INVALID', esc_html__( 'This order ID is invalid', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_ORDER_NUMBER_MISMATCH', esc_html__( 'Order Number mismatch', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_ORDER_PROCESSED', esc_html__( 'This order has already been processed', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_PDT_FAIL', esc_html__( 'PDT query failed', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_PDT_TOKEN_MISSING', esc_html__( 'PDT token not present in URL', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_SESSIONID_MISMATCH', esc_html__( 'Session ID mismatch', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_ERR_UNKNOWN', esc_html__( 'Unkown error occurred', 'woocommerce-gateway-smatpay' ) );

		// General messages.
		define( 'PF_MSG_OK', esc_html__( 'Payment was successful', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_MSG_FAILED', esc_html__( 'Payment has failed', 'woocommerce-gateway-smatpay' ) );
		define( 'PF_MSG_PENDING', esc_html__( 'The payment is pending. Please note, you will receive another Instant Transaction Notification when the payment status changes to "Completed", or "Failed"', 'woocommerce-gateway-smatpay' ) );

		/**
		 * Fires after Smatpay constants are setup.
		 *
		 * @since 1.4.13
		 */
		do_action( 'woocommerce_gateway_smatpay_setup_constants' );
	}

	/**
	 * Log system processes.
	 *
	 * @since 1.0.0
	 *
	 * @param string $message Log message.
	 */
	public function log( $message ) {
		if ( 'yes' === $this->get_option( 'testmode' ) || $this->enable_logging ) {
			if ( empty( $this->logger ) ) {
				$this->logger = new WC_Logger();
			}
			$this->logger->add( 'smatpay', $message );
		}
	}

	/**
	 * Validate the signature against the returned data.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $data      Returned data.
	 * @param string $signature Signature to check.
	 * @return bool Whether the signature is valid.
	 */
	public function validate_signature( $data, $signature ) {
		$result = $data['signature'] === $signature;
		$this->log( 'Signature = ' . ( $result ? 'valid' : 'invalid' ) );
		return $result;
	}

	/**
	 * Validate the IP address to make sure it's coming from Smatpay.
	 *
	 * @param string $source_ip Source IP.
	 * @since 1.0.0
	 * @return bool
	 */
	public function is_valid_ip( $source_ip ) {
		// Variable initialization.
		$valid_hosts = array(
			'www.smatpay.co.za',
			'sandbox.smatpay.co.za',
			'w1w.smatpay.co.za',
			'w2w.smatpay.co.za',
		);

		$valid_ips = array();

		foreach ( $valid_hosts as $pf_hostname ) {
			$ips = gethostbynamel( $pf_hostname );

			if ( false !== $ips ) {
				$valid_ips = array_merge( $valid_ips, $ips );
			}
		}

		// Remove duplicates.
		$valid_ips = array_unique( $valid_ips );

		// Adds support for X_Forwarded_For.
		if ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$x_forwarded_http_header = trim( current( preg_split( '/[,:]/', sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) ) ) );
			$source_ip               = rest_is_ip_address( $x_forwarded_http_header ) ? rest_is_ip_address( $x_forwarded_http_header ) : $source_ip;
		}

		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r -- used for logging.
		$this->log( "Valid IPs:\n" . print_r( $valid_ips, true ) );
		$is_valid_ip = in_array( $source_ip, $valid_ips, true );

		/**
		 * Filter whether Smatpay Gateway IP address is valid.
		 *
		 * @since 1.4.13
		 *
		 * @param bool $is_valid_ip Whether IP address is valid.
		 * @param bool $source_ip   Source IP.
		 */
		return apply_filters( 'woocommerce_gateway_smatpay_is_valid_ip', $is_valid_ip, $source_ip );
	}

	/**
	 * Validate response data.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $post_data POST data for original request.
	 * @param string $proxy     Address of proxy to use or NULL if no proxy.
	 * @return bool
	 */
	public function validate_response_data( $post_data, $proxy = null ) {
		$this->log( 'Host = ' . $this->validate_url );
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r -- used for logging.
		$this->log( 'Params = ' . print_r( $post_data, true ) );

		if ( ! is_array( $post_data ) ) {
			return false;
		}

		$response = wp_remote_post(
			$this->validate_url,
			array(
				'body'       => $post_data,
				'timeout'    => 70,
				'user-agent' => PF_USER_AGENT,
			)
		);

		if ( is_wp_error( $response ) || empty( $response['body'] ) ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r -- used for logging.
			$this->log( "Response error:\n" . print_r( $response, true ) );
			return false;
		}

		parse_str( $response['body'], $parsed_response );

		$response = $parsed_response;

		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r -- used for logging.
		$this->log( "Response:\n" . print_r( $response, true ) );

		// Interpret Response.
		if ( is_array( $response ) && in_array( 'VALID', array_keys( $response ), true ) ) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Check the given amounts are equal.
	 *
	 * Checks to see whether the given amounts are equal using a proper floating
	 * point comparison with an Epsilon which ensures that insignificant decimal
	 * places are ignored in the comparison.
	 *
	 * eg. 100.00 is equal to 100.0001
	 *
	 * @since 1.0.0
	 *
	 * @param float $amount1 1st amount for comparison.
	 * @param float $amount2 2nd amount for comparison.
	 *
	 * @return bool
	 */
	public function amounts_equal( $amount1, $amount2 ) {
		return ! ( abs( floatval( $amount1 ) - floatval( $amount2 ) ) > PF_EPSILON );
	}

	/**
	 * Get order property with compatibility check on order getter introduced
	 * in WC 3.0.
	 *
	 * @since 1.4.1
	 *
	 * @param WC_Order $order Order object.
	 * @param string   $prop  Property name.
	 *
	 * @return mixed Property value
	 */
	public static function get_order_prop( $order, $prop ) {
		switch ( $prop ) {
			case 'order_total':
				$getter = array( $order, 'get_total' );
				break;
			default:
				$getter = array( $order, 'get_' . $prop );
				break;
		}

		return is_callable( $getter ) ? call_user_func( $getter ) : $order->{ $prop };
	}

	/**
	 * Gets user-friendly error message strings from keys
	 *
	 * @param   string $key  The key representing an error.
	 *
	 * @return  string        The user-friendly error message for display
	 */
	public function get_error_message( $key ) {
		switch ( $key ) {
			case 'wc-gateway-smatpay-error-invalid-currency':
				return esc_html__( 'Your store uses a currency that Smatpay doesn\'t support yet.', 'woocommerce-gateway-smatpay' );
			case 'wc-gateway-smatpay-error-missing-merchant-id':
				return esc_html__( 'You forgot to fill your merchant ID.', 'woocommerce-gateway-smatpay' );
			case 'wc-gateway-smatpay-error-missing-merchant-key':
				return esc_html__( 'You forgot to fill your merchant key.', 'woocommerce-gateway-smatpay' );
			case 'wc-gateway-smatpay-error-missing-pass-phrase':
				return esc_html__( 'Smatpay requires a passphrase to work.', 'woocommerce-gateway-smatpay' );
			default:
				return '';
		}
	}

	/**
	 * Show possible admin notices
	 */
	public function admin_notices() {

		// Get requirement errors.
		$errors_to_show = $this->check_requirements();

		// If everything is in place, don't display it.
		if ( ! count( $errors_to_show ) ) {
			return;
		}

		// If the gateway isn't enabled, don't show it.
		if ( 'no' === $this->enabled ) {
			return;
		}

		// Use transients to display the admin notice once after saving values.
		if ( ! get_transient( 'wc-gateway-smatpay-admin-notice-transient' ) ) {
			set_transient( 'wc-gateway-smatpay-admin-notice-transient', 1, 1 );

			echo '<div class="notice notice-error is-dismissible"><p>'
				. esc_html__( 'To use Smatpay as a payment provider, you need to fix the problems below:', 'woocommerce-gateway-smatpay' ) . '</p>'
				. '<ul style="list-style-type: disc; list-style-position: inside; padding-left: 2em;">'
				. wp_kses_post(
					array_reduce(
						$errors_to_show,
						function ( $errors_list, $error_item ) {
							$errors_list = $errors_list . PHP_EOL . ( '<li>' . $this->get_error_message( $error_item ) . '</li>' );
							return $errors_list;
						},
						''
					)
				)
				. '</ul></p></div>';
		}
	}

	/**
	 * Displays the amount_fee as returned by smatpay.
	 *
	 * @param int $order_id The ID of the order.
	 */
	public function display_order_fee( $order_id ) {

		$order = wc_get_order( $order_id );
		$fee   = $order->get_meta( 'smatpay_amount_fee', true );

		if ( ! $fee ) {
			return;
		}
		?>

		<tr>
			<td class="label smatpay-fee">
				<?php echo wc_help_tip( __( 'This represents the fee Smatpay collects for the transaction.', 'woocommerce-gateway-smatpay' ) ); ?>
				<?php esc_html_e( 'Smatpay Fee:', 'woocommerce-gateway-smatpay' ); ?>
			</td>
			<td width="1%"></td>
			<td class="total">
				<?php echo wp_kses_post( wc_price( $fee, array( 'decimals' => 2 ) ) ); ?>
			</td>
		</tr>

		<?php
	}

	/**
	 * Displays the amount_net as returned by smatpay.
	 *
	 * @param int $order_id The ID of the order.
	 */
	public function display_order_net( $order_id ) {

		$order = wc_get_order( $order_id );
		$net   = $order->get_meta( 'smatpay_amount_net', true );

		if ( ! $net ) {
			return;
		}

		?>

		<tr>
			<td class="label smatpay-net">
				<?php echo wc_help_tip( __( 'This represents the net total that was credited to your Smatpay account.', 'woocommerce-gateway-smatpay' ) ); ?>
				<?php esc_html_e( 'Amount Net:', 'woocommerce-gateway-smatpay' ); ?>
			</td>
			<td width="1%"></td>
			<td class="total">
				<?php echo wp_kses_post( wc_price( $net, array( 'decimals' => 2 ) ) ); ?>
			</td>
		</tr>

		<?php
	}

	/**
	 * Filters the currency to 'ZAR' if set via WooPayments multi-currency feature.
	 *
	 * @param string $currency The currency code.
	 * @return string
	 */
	public function filter_currency( $currency ) {
		// Do nothing if WooPayments is not activated.
		if ( ! class_exists( '\WCPay\MultiCurrency\MultiCurrency' ) ) {
			return $currency;
		}

		// Do nothing if the page is admin screen.
		if ( is_admin() ) {
			return $currency;
		}

		$user_id = get_current_user_id();

		// Check if the currency is set in the URL.
		if ( isset( $_GET['currency'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$currency_code = sanitize_text_field(
				wp_unslash( $_GET['currency'] ) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			);
			// Check if the currency is set in the session (for logged-out users).
		} elseif ( 0 === $user_id && WC()->session ) {
			$currency_code = WC()->session->get( \WCPay\MultiCurrency\MultiCurrency::CURRENCY_SESSION_KEY );
			// Check if the currency is set in the user meta (for logged-in users).
		} elseif ( $user_id ) {
			$currency_code = get_user_meta( $user_id, \WCPay\MultiCurrency\MultiCurrency::CURRENCY_META_KEY, true );
		}

		if ( is_string( $currency_code ) && 'ZAR' === $currency_code ) {
			return 'ZAR';
		}

		return $currency;
	}
}
